/**
 * Copyright (c) 2016-2019 人人开源 All rights reserved.
 *
 * https://www.renren.io
 *
 * 版权所有，侵权必究！
 */

package tech.glinfo.enbao.common.aspect;

import com.alibaba.fastjson.JSON;
import tech.glinfo.enbao.common.annotation.ApiLog;
import tech.glinfo.enbao.common.utils.HttpContextUtils;
import tech.glinfo.enbao.common.utils.IPUtils;
import tech.glinfo.enbao.common.utils.JwtUtils;
import tech.glinfo.enbao.modules.sys.entity.AppApiLogEntity;
import tech.glinfo.enbao.modules.sys.service.AppApiLogService;
import com.google.gson.Gson;
import io.jsonwebtoken.Claims;
import org.apache.commons.lang.StringUtils;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import javax.servlet.http.HttpServletRequest;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;


/**
 * 系统日志，切面处理类
 *
 * @author Mark sunlightcs@gmail.com
 */
@Aspect
@Component
public class ApiLogAspect {
	protected Logger logger = LoggerFactory.getLogger(getClass());
	@Autowired
	private AppApiLogService appApiLogService;
	@Autowired
	private JwtUtils jwtUtils;
	@Value("${spring.profiles.active}")
	private String env;
	
	@Pointcut("@annotation(tech.glinfo.enbao.common.annotation.ApiLog)")
	public void logPointCut() { 
		
	}

	@Around("logPointCut()")
	public Object around(ProceedingJoinPoint point) throws Throwable {
		long beginTime = System.currentTimeMillis();
		//执行方法
		Object result = point.proceed();
		//执行时长(毫秒)
		long time = System.currentTimeMillis() - beginTime;

		//保存日志
		saveSysLog(point, time);

		if("test".equals(env)) {
			logger.info("result:{}", JSON.toJSONString(result));
		}

		return result;
	}

	private void saveSysLog(ProceedingJoinPoint joinPoint, long time) {
		MethodSignature signature = (MethodSignature) joinPoint.getSignature();
		Method method = signature.getMethod();

		AppApiLogEntity apiLog = new AppApiLogEntity();
		ApiLog syslog = method.getAnnotation(ApiLog.class);
		if(syslog != null){
			//注解上的描述
			apiLog.setOperation(syslog.value());
		}

		//请求的方法名
		String className = joinPoint.getTarget().getClass().getName();
		String methodName = signature.getName();
		apiLog.setMethod(className + "." + methodName + "()");

		//请求的参数
		Object[] args = joinPoint.getArgs();
		//过滤AppUser对象，不需要保存
		List<Object> realArgs = new ArrayList<>();
		try{
			String params = new Gson().toJson(realArgs);
			apiLog.setParams(params);
		}catch (Exception e){

		}

		//获取request
		HttpServletRequest request = HttpContextUtils.getHttpServletRequest();
		//设置IP地址
		apiLog.setIp(IPUtils.getIpAddr(request));

		//用户名
		//获取用户凭证
		String token = request.getHeader(jwtUtils.getHeader());
		if(StringUtils.isBlank(token)){
			token = request.getParameter(jwtUtils.getHeader());
		}

		//凭证为空
		if(StringUtils.isNotBlank(token)){
			Claims claims = jwtUtils.getClaimByToken(token);
			if(claims != null && !jwtUtils.isTokenExpired(claims.getExpiration())){
				String[] user = claims.getSubject().split("#");
				apiLog.setUserid(Integer.valueOf(user[0]));
				apiLog.setPhone(user[1]);
			}
		}


		apiLog.setTime(time);
		apiLog.setCreateDate(new Date());
		//保存系统日志
		appApiLogService.save(apiLog);
	}
}
