package tech.glinfo.enbao.modules.websocket;

import cn.hutool.log.Log;
import cn.hutool.log.LogFactory;
import io.jsonwebtoken.Claims;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import tech.glinfo.enbao.common.utils.JwtUtils;
import tech.glinfo.enbao.common.utils.StringUtils;

import javax.websocket.*;
import javax.websocket.server.PathParam;
import javax.websocket.server.ServerEndpoint;
import java.io.IOException;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * @author linyetong
 * @date 2020/2/10 20:48
 */
@ServerEndpoint("/ws/{token}/{deviceId}")
@Component
public class WebSocketServer {

    static Log log = LogFactory.get(WebSocketServer.class);
    /**
     * concurrent包的线程安全Set，用来存放每个客户端对应的MyWebSocket对象。
     */
    private static ConcurrentHashMap<String, Session> webSocketMap = new ConcurrentHashMap<>();
    /**
     * 与某个客户端的连接会话，需要通过它来给客户端发送数据
     */
    private Session session;
    /**
     * 接收userId
     */
    private String key = null;

    private static JwtUtils jwtUtils;

    @Autowired
    public void setJwtUtils(JwtUtils jwtUtils) {
        WebSocketServer.jwtUtils = jwtUtils;
    }

    /**
     * 连接建立成功调用的方法
     */
    @OnOpen
    public void onOpen(Session session, @PathParam("token") String token, @PathParam("deviceId") String deviceId) throws IOException {
        this.session = session;
        Claims claims = jwtUtils.getClaimByToken(token);
        if(claims == null || jwtUtils.isTokenExpired(claims.getExpiration())){
            log.error("token 不存在或已经过期:{}", token);
            session.close();
            return;
        }
        this.key = token+"#"+deviceId;
        if (webSocketMap.containsKey(key)) {
            webSocketMap.remove(key);
            webSocketMap.put(key, session);
            //加入set中
        } else {
            webSocketMap.put(key, session);
            //加入set中
        }

        log.info("用户连接:" + key + ",当前在线人数为:" + webSocketMap.size());

//        try {
//            sendMessage(session,"{\"connect\":\"OK\"}");
//        } catch (IOException e) {
//            log.error("用户:" + key + ",网络异常!!!!!!");
//        }
    }

    /**
     * 连接关闭调用的方法
     */
    @OnClose
    public void onClose() {
        if (webSocketMap.containsKey(key)) {
            webSocketMap.remove(key);
        }
        log.info("用户退出:" + key + ",当前在线人数为:" + webSocketMap.size());
    }

    /**
     * 收到客户端消息后调用的方法
     *
     * @param message 客户端发送过来的消息
     */
    @OnMessage
    public void onMessage(String message, Session session) {
        log.info("用户消息:" + key + ",报文:" + message);
        //可以群发消息
        //消息保存到数据库、redis
        /*if (StringUtils.isNotBlank(message)) {
            try {
                //解析发送的报文
                JSONObject jsonObject = JSON.parseObject(message);
                //追加发送人(防止串改)
                jsonObject.put("fromUserId", this.key);
                String toUserId = jsonObject.getString("toUserId");
                //传送给对应toUserId用户的websocket
                if (StringUtils.isNotBlank(toUserId) && webSocketMap.containsKey(toUserId)) {
                    webSocketMap.get(toUserId).sendMessage(jsonObject.toJSONString());
                } else {
                    log.error("请求的userId:" + toUserId + "不在该服务器上");
                    //否则不在这个服务器上，发送到mysql或者redis
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }*/
    }

    /**
     * @param session
     * @param error
     */
    @OnError
    public void onError(Session session, Throwable error) {
        log.error("出错了,用户:{}, 原因：{}", this.key, error.getMessage());
//        log.error(error);
    }

    /**
     * 实现服务器主动推送
     */
    public static void sendMessage(Session session, String message) throws IOException {
        session.getBasicRemote().sendText(message);
    }

    /**
     * 发送自定义消息
     */
    public static void sendByDeviceId(String message, String deviceId) throws IOException {
        log.info("发送消息到:" + deviceId + "，报文:" + message);
        if(StringUtils.isBlank(deviceId)) {
            return;
        }
        //遍历这台机器上的所有websocket，给指定设备发送消息
        Iterator<Map.Entry<String, Session>> it = webSocketMap.entrySet().iterator();
        boolean isExist = false;
        while (it.hasNext()) {
            Map.Entry<String, Session> next = it.next();
            String key = next.getKey();
            String[] keys = key.split("#");
            if(deviceId.equals(keys[1])) {
                sendMessage(next.getValue(), message);
                isExist = true;
            }
        }
        if(!isExist) {
            log.error("设备:{} 不在线！", deviceId);
        }
    }

    /**
     * 发送自定义消息
     */
    public static void sendAll(String message) throws IOException {
        log.info("发送消息到所有设备，报文:" + message);
        //遍历这台机器上的所有websocket，给指定设备发送消息
        Iterator<Map.Entry<String, Session>> it = webSocketMap.entrySet().iterator();
        boolean isExist = false;
        while (it.hasNext()) {
            Map.Entry<String, Session> next = it.next();
            sendMessage(next.getValue(), message);
        }
        log.info("发送完毕，总共有：{} 台设备！", webSocketMap.size());
    }

}
